/** @file   LightballAlien.cpp
 * @brief   Implementation of LightballAlien class.
 * @version $Revision: 1.3 $
 * @date    $Date: 2006/06/21 16:08:38 $
 * @author  Tomi Lamminsaari
 */

#include "LightballAlien.h"
#include "gameanims.h"
#include "animplayer.h"
#include "LightballAi.h"
#include "settings.h"
#include "warglobals.h"
#include "soundsamples.h"
#include "AnimId.h"
using namespace eng2d;

namespace WeWantWar {

///
/// Constants, datatypes and static methods
/// ============================================================================

const int LightballAlien::GUNNOSE_CONTROLPOINT;


///
/// Constructors, destructor and operators
/// ============================================================================

/** Default constructor.
 */
LightballAlien::LightballAlien() :
  Alien()
{
  int oid = ObjectID::TYPE_LIGHTBALLALIEN;
  
  // Set default properties
  iWalkVec.vy = -Settings::floatObjProp(oid, "spd_walk:");
  this->setCorrectAnimation( GameAnims::EIdle );
  this->boundingSphere( Settings::floatObjProp(oid, "bounding_sphere:") );
  this->setArmor( Settings::floatObjProp(oid, "armor:") );
  iReloadingDelay = Settings::intObjProp(oid, "reload:");
  
  // Set the collision points
  this->setCollisionPoint( 0, Vec2D( -13,-13 ) );
  this->setCollisionPoint( 1, Vec2D(  13,-13 ) );
  this->setCollisionPoint( 2, Vec2D(  13, 13 ) );
  this->setCollisionPoint( 3, Vec2D( -13, 13 ) );
  
  // Set the controlpoints
  this->addCtrlPoint( Vec2D(11,-28) );
  
  // Create the controller
  this->setController( new LightballAi(this) );
}



/** Destructor.
 */
LightballAlien::~LightballAlien()
{
}



///
/// Methods inhertited from the base class(es)
/// ============================================================================

/** Updates this alien.
 */
void LightballAlien::update()
{
  if ( this->state() == GameObject::STATE_KILLED ) {
    return;
  }
  
  if ( this->state() == GameObject::STATE_DYING ) {
    if ( m_animation.paused() == true ) {
      // The dying animation has been played through. We mark this alien
      // as killed.
      this->state( GameObject::STATE_KILLED );
      this->setCorrectAnimation( GameAnims::EKilled );
    }
    return;
  }
  
  m_pController->update();
  
  if ( m_pController->forward() != 0 ) {
    // We're walking
    Vec2D mvec( iWalkVec );
    mvec.rotate( this->angle() );
    this->move( mvec );
    this->setCorrectAnimation( GameAnims::EWalk );
  }
  if ( m_pController->left() != 0 ) {
    Vec2D mvec( iWalkVec );
    mvec *= 0.7;
    mvec.rotate( this->angle() );
    mvec.rotate( 224 );
    this->move( mvec );
    
  } else if ( m_pController->right() != 0 ) {
    Vec2D mvec( iWalkVec );
    mvec *= 0.7;
    mvec.rotate( 64 );
    this->move( mvec );
    
  }
  
  this->changeAngle( m_pController->turn() );
  
  if ( m_pController->shoot() != 0 ) {
    // We should shoot
    this->attack();
  }
  
  if ( m_pController->forward() == 0 && m_pController->shoot() == 0 ) {
    this->setCorrectAnimation( GameAnims::EIdle );
  }
}



/** Makes sound.
 */
void LightballAlien::makeSound( GameObject::SoundID aSoundId ) const
{
  int sampleIndex = -1;
  switch ( aSoundId ) {
    case ( GameObject::SND_PAIN ): {
      sampleIndex = SMP_ALIEN1_PAIN;
      break;
    }
    case ( GameObject::SND_DIE ): {
      sampleIndex = SMP_ALIEN1_DIE;
      break;
    }
    case ( GameObject::SND_ATTACK ): {
      sampleIndex = SMP_LIGHTBALLGUN;
      break;
    }
    default: {
      sampleIndex = -1;
      break;
    }
  }
  if ( sampleIndex != -1 ) {
    Sound::playSample( sampleIndex, false );
  }
}



/** Handles the bullet hits
 */
bool LightballAlien::hitByBullet( Bullet* aBullet )
{
  bool ret = Alien::hitByBullet( aBullet );
  if ( ret == true ) {
    if ( aBullet->iType != Bullet::EFlameThrower ) {
      ParticleBlood* pP = new ParticleBlood( aBullet->iPosition,
                                             aBullet->velocity(), 12,
                                             Color(130,60,10) );
      WarGlobals::pPartManager->addSystem( pP );
    }
  }

  // Inform the AI-controller that we got hit.
  AIController* pC = dynamic_cast<AIController*>( this->getController() );
  AIController::BFlag f = pC->getFlags();
  f &= ~AIController::WAIT_TARGET;
  pC->setFlags( f );
  return ret;
}



/** Kills this object.
 */
void LightballAlien::kill()
{
  this->state( GameObject::STATE_DYING );
  this->setCorrectAnimation( GameAnims::EDying );
  this->makeSound( GameObject::SND_DIE );
}



/** Returns the type of this object.
 */
ObjectID::Type LightballAlien::objectType() const
{
  return ObjectID::TYPE_LIGHTBALLALIEN;
}




///
/// New public methods
/// ============================================================================




///
/// Getter methods
/// ============================================================================




///
/// Protected and private methods
/// ============================================================================

/** Sets the requested animation. 
 */
void LightballAlien::setCorrectAnimation( int aAnimId )
{
  if ( this->getAnimID() != aAnimId ) {
    const Animation& anim = GameAnims::findAnimation( AnimId::KLightballAlien, aAnimId );
    this->setAnimation( anim, aAnimId );
  }
}



/** Shoots
 */
void LightballAlien::attack()
{
  if ( this->reloading() == true ) {
    if ( m_pController->forward() == 1 ) {
      this->setCorrectAnimation( GameAnims::EShootWalk );
    }
    return;
  }
  
  Vec2D gunV( this->getCtrlPoint( GUNNOSE_CONTROLPOINT ) );
  gunV += this->position();
  Bullet* bullet = BulletTable::createBullet(this, gunV, Bullet::EAlienFireball);
  WarGlobals::pBulletManager->spawnBullet( bullet );
  
  const Animation& lightAnim = GameAnims::findAnimation( AnimId::KRifleShotLight );
  if ( Settings::shootingLights == true ) {
    AnimPlayer::spawn( lightAnim, gunV, 0 );
  }
  this->makeSound( GameObject::SND_ATTACK );
  this->setCounter( Alien::RELOAD_COUNTER_INDEX, iReloadingDelay );
  if ( m_pController->forward() == 0 ) {
    this->setCorrectAnimation( GameAnims::EShoot );
  } else {
    this->setCorrectAnimation( GameAnims::EShootWalk );
  }
}


};  // end of namespace
